const fileInput = document.getElementById("file-input");
const dropZone = document.getElementById("drop-zone");
const convertBtn = document.getElementById("convert-btn");
const output = document.getElementById("output");
const progress = document.getElementById("progress");
const qualityRange = document.getElementById("quality");
const qualityVal = document.getElementById("quality-val");

qualityRange.addEventListener("input", () => {
  qualityVal.textContent = qualityRange.value + "%";
});

dropZone.addEventListener("dragover", (e) => {
  e.preventDefault();
  dropZone.classList.add("highlight");
});
dropZone.addEventListener("dragleave", () => dropZone.classList.remove("highlight"));
dropZone.addEventListener("drop", (e) => {
  e.preventDefault();
  dropZone.classList.remove("highlight");
  fileInput.files = e.dataTransfer.files;
});

function parsePageRange(range, totalPages) {
  if (!range) return [...Array(totalPages).keys()].map((i) => i + 1);
  return range
    .split(",")
    .flatMap((part) => {
      if (part.includes("-")) {
        const [start, end] = part.split("-").map(Number);
        return Array.from({ length: end - start + 1 }, (_, i) => start + i);
      }
      return [Number(part)];
    })
    .filter((n) => n >= 1 && n <= totalPages);
}

convertBtn.addEventListener("click", async () => {
  output.innerHTML = "";
  progress.textContent = "";

  const files = fileInput.files;
  const format = document.getElementById("format").value;
  const quality = +document.getElementById("quality").value / 100;
  const dpi = +document.getElementById("dpi").value;
  const grayscale = document.getElementById("grayscale").checked;
  const pageRange = document.getElementById("page-range").value;

  for (const file of files) {
    const buffer = await file.arrayBuffer();
    const pdf = await pdfjsLib.getDocument({ data: buffer }).promise;
    const pages = parsePageRange(pageRange, pdf.numPages);

    for (let i = 0; i < pages.length; i++) {
      const num = pages[i];
      progress.textContent = `Processing: ${file.name} (Page ${num} of ${pdf.numPages})...`;

      const page = await pdf.getPage(num);
      const scale = dpi / 72;
      const viewport = page.getViewport({ scale });
      const canvas = document.createElement("canvas");
      const ctx = canvas.getContext("2d");
      canvas.width = viewport.width;
      canvas.height = viewport.height;

      await page.render({ canvasContext: ctx, viewport }).promise;

      if (grayscale) {
        const imgData = ctx.getImageData(0, 0, canvas.width, canvas.height);
        for (let j = 0; j < imgData.data.length; j += 4) {
          const avg = (imgData.data[j] + imgData.data[j + 1] + imgData.data[j + 2]) / 3;
          imgData.data[j] = imgData.data[j + 1] = imgData.data[j + 2] = avg;
        }
        ctx.putImageData(imgData, 0, 0);
      }

      const imgURL = canvas.toDataURL(`image/${format}`, quality);
      const img = new Image();
      img.src = imgURL;
      output.appendChild(img);

      const link = document.createElement("a");
      link.href = imgURL;
      link.download = `${file.name.replace(/\.pdf$/i, "")}-page${num}.${format}`;
      link.textContent = `Download Page ${num}`;
      output.appendChild(link);
    }
  }

  progress.textContent = "✅ All done!";
});
